/* ***************************************************** **
   ch21_bin_fitting.sql
   
   Skrypt dla książki Praktyczna nauka SQL dla Oracle, Helion (2022),
   napisanej przez Kima Berga Hansena, https://www.kibeha.dk
   Używasz na własną odpowiedzialność.
   *****************************************************
   
   Rozdział 21.
   Optymalizacja pakowania
   
   Skrypt przeznaczony do wykonania w schemacie PRACTICAL
** ***************************************************** */

/* -----------------------------------------------------
   Konfiguracja formatowania sqlcl
   ----------------------------------------------------- */

set pagesize 80
set linesize 80
set sqlformat ansiconsole

alter session set nls_date_format = 'YYYY-MM-DD';

/* -----------------------------------------------------
   Przykładowy kod do rozdziału 21.
   ----------------------------------------------------- */

-- Listing 21.1. Zapytanie wyświetlające zapasy magazynowe dotyczące jednego z gatunków piwa

select
   product_name
 , warehouse as wh
 , aisle
 , position  as pos
 , qty
from inventory_with_dims
where product_name = 'Der Helle Kumpel'
order by wh, aisle, pos;

-- Optymalizacja pakowania za pomocą nieograniczonej liczby kartonów o ograniczonej pojemności

-- Listing 21.2. Optymalizacja pakowania prowadzona w kolejności położenia butelek piwa w magazynie

select wh, aisle, pos, qty, run_qty, box#, box_qty
from (
   select
      product_name
    , warehouse as wh
    , aisle
    , position  as pos
    , qty
   from inventory_with_dims
   where product_name = 'Der Helle Kumpel'
) iwd
match_recognize (
   order by wh, aisle, pos
   measures
      match_number()   as box#
    , running sum(qty) as run_qty
    , final   sum(qty) as box_qty
   all rows per match
   pattern (
      fits_in_box+
   )
   define
      fits_in_box as sum(qty) <= 72
)
order by wh, aisle, pos;

-- Zmiana kolejności na malejącą według ilości

select wh, aisle, pos, qty, run_qty, box#, box_qty
from (
   select
      product_name
    , warehouse as wh
    , aisle
    , position  as pos
    , qty
   from inventory_with_dims
   where product_name = 'Der Helle Kumpel'
) iwd
match_recognize (
   order by qty desc, wh, aisle, pos
   measures
      match_number()   as box#
    , running sum(qty) as run_qty
    , final   sum(qty) as box_qty
   all rows per match
   pattern (
      fits_in_box+
   )
   define
      fits_in_box as sum(qty) <= 72
)
order by qty desc, wh, aisle, pos;

-- Listing 21.3. Używanie prostego algorytmu przybliżonego najlepszego dopasowania

select wh, aisle, pos, qty, run_qty, box#, box_qty
     , prio ,rn
from (
   select
      product_name
    , warehouse as wh
    , aisle
    , position  as pos
    , qty
    , case when qty > 72*2/3 then 1 else 2 end prio
    , least(
         row_number() over (
            partition by
               case when qty > 72*2/3 then 1 else 2 end
            order by qty
         )
       , row_number() over (
            partition by
               case when qty > 72*2/3 then 1 else 2 end
            order by qty desc
         )
      ) rn
   from inventory_with_dims
   where product_name = 'Der Helle Kumpel'
) iwd
match_recognize (
   order by prio, rn, qty desc, wh, aisle, pos
   measures
      match_number()   as box#
    , running sum(qty) as run_qty
    , final   sum(qty) as box_qty
   all rows per match
   pattern (
      fits_in_box+
   )
   define
      fits_in_box as sum(qty) <= 72
)
order by prio, rn, qty desc, wh, aisle, pos;

-- Listing 21.4. Wykorzystanie partycjonowania w algorytmie optymalizacji pakowania

select product_id
     , wh, aisle, pos, qty, run_qty, box#, box_qty
from (
   select
      product_id
    , product_name
    , warehouse as wh
    , aisle
    , position  as pos
    , qty
    , case when qty > 72*2/3 then 1 else 2 end prio
    , least(
         row_number() over (
            partition by
               product_id
             , case when qty > 72*2/3 then 1 else 2 end
            order by qty
         )
       , row_number() over (
            partition by
               product_id
             , case when qty > 72*2/3 then 1 else 2 end
            order by qty desc
         )
      ) rn
   from inventory_with_dims
) iwd
match_recognize (
   partition by product_id
   order by prio, rn, qty desc, wh, aisle, pos
   measures
      match_number()   as box#
    , running sum(qty) as run_qty
    , final   sum(qty) as box_qty
   all rows per match
   pattern (
      fits_in_box+
   )
   define
      fits_in_box as sum(qty) <= 72
)
order by product_id, prio, rn, qty desc, wh, aisle, pos;

-- Listing 21.5. Pobieranie pojedynczego rekordu danych wyjściowych dla każdego kartonu

select product_id, product_name, box#, box_qty, locs
from (
   select
      product_id
    , product_name
    , warehouse as wh
    , aisle
    , position  as pos
    , qty
    , case when qty > 72*2/3 then 1 else 2 end prio
    , least(
         row_number() over (
            partition by
               product_id
             , case when qty > 72*2/3 then 1 else 2 end
            order by qty
         )
       , row_number() over (
            partition by
               product_id
             , case when qty > 72*2/3 then 1 else 2 end
            order by qty desc
         )
      ) rn
   from inventory_with_dims
) iwd
match_recognize (
   partition by product_id
   order by prio, rn, qty desc, wh, aisle, pos
   measures
      max(product_name) as product_name
    , match_number()    as box#
    , final sum(qty)    as box_qty
    , final count(*)    as locs
   one row per match
   pattern (
      fits_in_box+
   )
   define
      fits_in_box as sum(qty) <= 72
)
order by product_id, box#;

-- Listing 21.6. Problemy pojawiające się podczas pakowania produktów do zbyt małych kartonów

select wh, aisle, pos, qty, run_qty, box#, box_qty
from (
   select
      product_name
    , warehouse as wh
    , aisle
    , position  as pos
    , qty
   from inventory_with_dims
   where product_name = 'Der Helle Kumpel'
) iwd
match_recognize (
   order by wh, aisle, pos
   measures
      match_number()   as box#
    , running sum(qty) as run_qty
    , final   sum(qty) as box_qty
   all rows per match
   pattern (
      fits_in_box+
   )
   define
      fits_in_box as sum(qty) <= 64
)
order by wh, aisle, pos;

-- Zmiana znaku + we wzorcu na znak *

select wh, aisle, pos, qty, run_qty, box#, box_qty
from (
   select
      product_name
    , warehouse as wh
    , aisle
    , position  as pos
    , qty
   from inventory_with_dims
   where product_name = 'Der Helle Kumpel'
) iwd
match_recognize (
   order by wh, aisle, pos
   measures
      match_number()   as box#
    , running sum(qty) as run_qty
    , final   sum(qty) as box_qty
   all rows per match
   pattern (
      fits_in_box*
   )
   define
      fits_in_box as sum(qty) <= 64
)
order by wh, aisle, pos;

-- Wykluczenie pustych dopasowań

select wh, aisle, pos, qty, run_qty, box#, box_qty
from (
   select
      product_name
    , warehouse as wh
    , aisle
    , position  as pos
    , qty
   from inventory_with_dims
   where product_name = 'Der Helle Kumpel'
) iwd
match_recognize (
   order by wh, aisle, pos
   measures
      match_number()   as box#
    , running sum(qty) as run_qty
    , final   sum(qty) as box_qty
   all rows per match omit empty matches
   pattern (
      fits_in_box*
   )
   define
      fits_in_box as sum(qty) <= 64
)
order by wh, aisle, pos;

-- Przywrócenie znaku + we wzorcu, ale z niedopasowanymi rekordami

select wh, aisle, pos, qty, run_qty, box#, box_qty
from (
   select
      product_name
    , warehouse as wh
    , aisle
    , position  as pos
    , qty
   from inventory_with_dims
   where product_name = 'Der Helle Kumpel'
) iwd
match_recognize (
   order by wh, aisle, pos
   measures
      match_number()   as box#
    , running sum(qty) as run_qty
    , final   sum(qty) as box_qty
   all rows per match with unmatched rows
   pattern (
      fits_in_box+
   )
   define
      fits_in_box as sum(qty) <= 64
)
order by wh, aisle, pos;

-- Optymalizacja pakowania za pomocą ograniczonej liczby kartonów o nieograniczonej pojemności

-- Listing 21.7. Zapytanie wyświetlające butelki piwa Der Helle Kumpel według jego malejącej ilości w poszczególnych miejscach w magazynie

select
   product_name
 , warehouse as wh
 , aisle
 , position  as pos
 , qty
from inventory_with_dims
where product_name = 'Der Helle Kumpel'
order by qty desc, wh, aisle, pos;

-- Listing 21.8. Wszystkie rekordy w pojedynczym dopasowaniu, rozkładane za pomocą logiki w klauzuli define

select wh, aisle, pos, qty, box, qty1, qty2, qty3
from (
   select
      product_name
    , warehouse as wh
    , aisle
    , position  as pos
    , qty
   from inventory_with_dims
   where product_name = 'Der Helle Kumpel'
) iwd
match_recognize (
   order by qty desc, wh, aisle, pos
   measures
      classifier()          as box
    , running sum(box1.qty) as qty1
    , running sum(box2.qty) as qty2
    , running sum(box3.qty) as qty3
   all rows per match
   pattern (
      (box1 | box2 | box3)*
   )
   define
      box1 as count(box1.*) = 1
           or sum(box1.qty) - box1.qty
                <= least(sum(box2.qty), sum(box3.qty))
    , box2 as count(box2.*) = 1
           or sum(box2.qty) - box2.qty
                <= sum(box3.qty)
)
order by qty desc, wh, aisle, pos;

-- Listing 21.9. Wszystkie produkty w trzech kartonach — dane wyjściowe zostały posortowane według położenia produktów w magazynie

select product_name, wh, aisle, pos, qty, box
from (
   select
      product_id
    , product_name
    , warehouse as wh
    , aisle
    , position  as pos
    , qty
   from inventory_with_dims
) iwd
match_recognize (
   partition by product_id
   order by qty desc, wh, aisle, pos
   measures
      classifier()          as box
   all rows per match
   pattern (
      (box1 | box2 | box3)*
   )
   define
      box1 as count(box1.*) = 1
           or sum(box1.qty) - box1.qty
                <= least(sum(box2.qty), sum(box3.qty))
    , box2 as count(box2.*) = 1
           or sum(box2.qty) - box2.qty
                <= sum(box3.qty)
)
order by wh, aisle, pos;

/* ***************************************************** */
